<?php

namespace Database\Seeders;

use App\Enums\FlatStatus;
use App\Enums\UserRole;
use App\Enums\VehicleType;
use App\Enums\VisitorStatus;
use App\Enums\VisitorType;
use App\Models\Flat;
use App\Models\Guard;
use App\Models\Owner;
use App\Models\PatrollingLocation;
use App\Models\User;
use App\Models\Vehicle;
use App\Models\Visitor;
use App\Models\Wing;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;

class QuickTestSeeder extends Seeder
{
    /**
     * Quick seeder for testing basic functionality.
     */
    public function run(): void
    {
        $this->command->info('🚀 Quick Test Seeding...');

        // Create a wing
        $wing = Wing::factory()->create([
            'name' => 'A Wing',
            'no_floors' => 10,
            'no_flats' => 40,
            'no_lifts' => 2,
        ]);

        // Create an owner
        $owner = Owner::factory()->create([
            'name' => 'John Doe',
            'email' => 'john@example.com',
        ]);

        // Create a flat
        $flat = Flat::factory()->create([
            'wing_id' => $wing->id,
            'owner_id' => $owner->id,
            'flat_no' => 'A101',
            'status' => FlatStatus::ACTIVE,
        ]);

        // Create admin user
        User::factory()->create([
            'name' => 'Admin',
            'email' => 'admin@test.com',
            'password' => Hash::make('password'),
            'role' => UserRole::ADMIN,
            'owner_id' => null,
            'flat_id' => null,
        ]);

        // Create owner user
        $ownerUser = User::factory()->create([
            'name' => 'John Doe',
            'email' => 'john@example.com',
            'password' => Hash::make('password'),
            'role' => UserRole::OWNER,
            'owner_id' => $owner->id,
            'flat_id' => $flat->id,
            'relation' => 'self',
        ]);

        // Create a guard
        $guard = Guard::factory()->create([
            'wing_id' => $wing->id,
            'name' => 'Security Guard',
            'email' => 'guard@test.com',
        ]);

        // Create a vehicle
        Vehicle::factory()->create([
            'owner_id' => $owner->id,
            'flat_id' => $flat->id,
            'vehicle_type' => VehicleType::FOUR_WHEELER,
            'make' => 'Maruti',
            'model' => 'Swift',
            'registration_no' => 'MH12AB1234',
        ]);

        // Create patrolling locations
        PatrollingLocation::factory()->count(5)->create([
            'wing_id' => $wing->id,
            'is_active' => true,
        ]);

        // Create visitors in different states
        Visitor::factory()->create([
            'wing_id' => $wing->id,
            'flat_id' => $flat->id,
            'name' => 'Jane Smith',
            'phone' => '9876543210',
            'visitor_type' => VisitorType::GUEST,
            'purpose' => 'Personal visit',
            'status' => VisitorStatus::PENDING,
        ]);

        Visitor::factory()->create([
            'wing_id' => $wing->id,
            'flat_id' => $flat->id,
            'name' => 'Bob Wilson',
            'phone' => '9876543211',
            'visitor_type' => VisitorType::DELIVERY,
            'purpose' => 'Package delivery',
            'status' => VisitorStatus::APPROVED,
            'approved_by' => $ownerUser->id,
            'approved_at' => now(),
        ]);

        $this->command->info('✅ Quick test seeding completed!');
        $this->command->info('📧 Test users:');
        $this->command->info('   Admin: admin@test.com / password');
        $this->command->info('   Owner: john@example.com / password');
    }
}